# pix_config.py
from __future__ import annotations
import os
import sqlite3
import tkinter as tk
from tkinter import ttk, messagebox, filedialog
from typing import Callable, Dict

# ----------------- Persistência (tabela settings) -----------------
def _ensure_settings(conn: sqlite3.Connection):
    conn.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)

def _get_setting(conn: sqlite3.Connection, key: str, default: str = "") -> str:
    _ensure_settings(conn)
    row = conn.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return row[0] if row and row[0] is not None else default

def _set_settings(conn: sqlite3.Connection, data: Dict[str, str]) -> None:
    _ensure_settings(conn)
    for k, v in data.items():
        conn.execute(
            "INSERT INTO settings(key, value) VALUES(?, ?) "
            "ON CONFLICT(key) DO UPDATE SET value=excluded.value",
            (k, v if v is not None else "")
        )

# ----------------- PSPs suportados -----------------
PSP_LIST = [
    ("simulado",      "Simulado (sem confirmação automática)"),
    ("mercadopago",   "Mercado Pago"),
    ("efi",           "Efi / Gerencianet"),
    ("pagseguro",     "PagSeguro"),
    ("bb",            "Banco do Brasil"),
    ("itau",          "Itaú"),
    ("bradesco",      "Bradesco"),
    ("santander",     "Santander"),
    ("sicoob",        "Sicoob"),
    ("sicredi",       "Sicredi"),
    ("custom",        "Outro / Custom"),
]

PSP_DEFAULTS = {
    "mercadopago": {"PIX_API_BASE": "https://api.mercadopago.com"},
    "efi":         {"PIX_API_BASE": "https://api.efi.com.br"},
    "pagseguro":   {"PIX_API_BASE": "https://api.pagseguro.com"},
    "bb":          {"PIX_API_BASE": "https://api.bb.com.br"},
    "itau":        {"PIX_API_BASE": "https://pix.itau.com.br"},
    "bradesco":    {"PIX_API_BASE": "https://pix.bradesco.com.br"},
    "santander":   {"PIX_API_BASE": "https://trustly.santander.com.br"},
    "sicoob":      {"PIX_API_BASE": "https://api.sicoob.com.br"},
    "sicredi":     {"PIX_API_BASE": "https://api.sicredi.com.br"},
}

CONFIRM_MODES = [
    ("none",    "Nenhum (manual)"),
    ("polling", "Polling (consulta periódica)"),
    ("webhook", "Webhook (notificação)"),
    ("auto",    "Auto (polling + webhook)"),
]

# ----------------- UI -----------------
def open_pix_config(app, get_conn: Callable):
    root = app.root
    win = tk.Toplevel(root)
    win.title("Configuração PIX")
    win.geometry("760x640")
    win.transient(root)
    win.grab_set()
    win.focus_force()

    frm = ttk.Frame(win, padding=12)
    frm.pack(fill="both", expand=True)

    # Vars
    v_provider = tk.StringVar(value="simulado")
    v_key      = tk.StringVar()
    v_name     = tk.StringVar(value="FabricaDigital.shop")
    v_city     = tk.StringVar(value="SAO PAULO")
    v_api_base = tk.StringVar()
    v_client_id = tk.StringVar()
    v_client_secret = tk.StringVar()   # Access Token (MP) ou secret do PSP
    v_cert_path = tk.StringVar()
    v_webhook_url = tk.StringVar()
    v_txid_prefix = tk.StringVar(value="FD")
    v_poll_s   = tk.StringVar(value="3")
    v_confirm  = tk.StringVar(value="none")
    v_webhook_port = tk.StringVar(value="5001")
    v_webhook_token = tk.StringVar(value="")

    # Carregar existentes
    with get_conn() as conn:
        v_provider.set(_get_setting(conn, "PIX_PROVIDER", "simulado"))
        v_key.set(_get_setting(conn, "PIX_KEY", ""))
        v_name.set(_get_setting(conn, "PIX_MERCHANT_NAME", v_name.get()))
        v_city.set(_get_setting(conn, "PIX_MERCHANT_CITY", v_city.get()))
        v_api_base.set(_get_setting(conn, "PIX_API_BASE", PSP_DEFAULTS.get(v_provider.get(), {}).get("PIX_API_BASE", "")))
        v_client_id.set(_get_setting(conn, "PIX_CLIENT_ID", ""))
        v_client_secret.set(_get_setting(conn, "PIX_CLIENT_SECRET", ""))
        v_cert_path.set(_get_setting(conn, "PIX_CERT_PATH", ""))
        v_webhook_url.set(_get_setting(conn, "PIX_WEBHOOK_URL", ""))
        v_txid_prefix.set(_get_setting(conn, "PIX_TXID_PREFIX", v_txid_prefix.get()))
        v_poll_s.set(_get_setting(conn, "PIX_POLL_INTERVAL", v_poll_s.get()))
        v_confirm.set(_get_setting(conn, "PIX_CONFIRM_MODE", v_confirm.get()))
        v_webhook_port.set(_get_setting(conn, "PIX_WEBHOOK_PORT", v_webhook_port.get()))
        v_webhook_token.set(_get_setting(conn, "PIX_WEBHOOK_TOKEN", v_webhook_token.get()))

    row = 0
    ttk.Label(frm, text="Provedor (PSP):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    cbo = ttk.Combobox(frm, textvariable=v_provider, state="readonly",
                       values=[f"{c} — {l}" for c, l in PSP_LIST], width=44)
    def _provider_code_only(*_):
        val = v_provider.get()
        if "—" in val:
            v_provider.set(val.split("—", 1)[0].strip())
    cbo.bind("<<ComboboxSelected>>", _provider_code_only)
    cbo.grid(row=row, column=1, sticky="w")
    row += 1

    ttk.Label(frm, text="Chave PIX:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ttk.Entry(frm, textvariable=v_key, width=48).grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Nome recebedor:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ttk.Entry(frm, textvariable=v_name, width=48).grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Cidade (UF opc.):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ttk.Entry(frm, textvariable=v_city, width=48).grid(row=row, column=1, sticky="w"); row += 1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="we", pady=(8,8)); row += 1
    ttk.Label(frm, text="Integração com PSP (opcional)").grid(row=row, column=0, columnspan=2, sticky="w"); row += 1

    ttk.Label(frm, text="API Base:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_api = ttk.Entry(frm, textvariable=v_api_base, width=48); ent_api.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Client ID:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_cid = ttk.Entry(frm, textvariable=v_client_id, width=48); ent_cid.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Client Secret / Access Token:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_sec = ttk.Entry(frm, textvariable=v_client_secret, show="•", width=48); ent_sec.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Certificado (PEM/P12):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    fcert = ttk.Frame(frm); fcert.grid(row=row, column=1, sticky="we")
    ttk.Entry(fcert, textvariable=v_cert_path, width=38).pack(side="left", fill="x", expand=True)
    ttk.Button(fcert, text="Procurar…", command=lambda: _pick_file(v_cert_path)).pack(side="left", padx=(6,0)); row += 1

    ttk.Label(frm, text="Webhook URL pública (opcional):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_weburl = ttk.Entry(frm, textvariable=v_webhook_url, width=48); ent_weburl.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Prefixo do TXID:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_txid = ttk.Entry(frm, textvariable=v_txid_prefix, width=12); ent_txid.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Intervalo de polling (s):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_poll = ttk.Entry(frm, textvariable=v_poll_s, width=12); ent_poll.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="we", pady=(8,8)); row += 1
    ttk.Label(frm, text="Confirmação automática").grid(row=row, column=0, columnspan=2, sticky="w"); row += 1

    ttk.Label(frm, text="Modo:").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    cbo_mode = ttk.Combobox(frm, textvariable=v_confirm, state="readonly",
                            values=[f"{c} — {l}" for c, l in CONFIRM_MODES], width=44)
    def _mode_code_only(*_):
        val = v_confirm.get()
        if "—" in val:
            v_confirm.set(val.split("—", 1)[0].strip())
    cbo_mode.bind("<<ComboboxSelected>>", _mode_code_only)
    cbo_mode.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Webhook (porta local):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_wport = ttk.Entry(frm, textvariable=v_webhook_port, width=12); ent_wport.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Label(frm, text="Webhook (token opcional):").grid(row=row, column=0, sticky="e", pady=4, padx=(0,6))
    ent_wtok = ttk.Entry(frm, textvariable=v_webhook_token, width=24); ent_wtok.grid(row=row, column=1, sticky="w"); row += 1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="we", pady=(8,8)); row += 1

    info = tk.Text(frm, height=6, width=74, wrap="word")
    info.insert("1.0",
        "Observações:\n"
        "• Mercado Pago: use o Access Token em “Client Secret / Access Token”.\n"
        "• Modo Webhook precisa que seu PSP consiga chamar seu computador. Se ele não for público, use um túnel (ex.: ngrok) e informe a URL em “Webhook URL pública”.\n"
        "• Auto = Webhook + Polling; o primeiro que confirmar vence.\n"
    )
    info.configure(state="disabled")
    info.grid(row=row, column=0, columnspan=2, sticky="we"); row += 1

    btns = ttk.Frame(frm); btns.grid(row=row, column=0, columnspan=2, pady=(10,0), sticky="e")
    ttk.Button(btns, text="Salvar", command=lambda: _save()).pack(side="left")
    ttk.Button(btns, text="Fechar", command=win.destroy).pack(side="left", padx=6)

    def _update_fields(*_):
        _provider_code_only(); _mode_code_only()
        code = v_provider.get()
        if not v_api_base.get():
            v_api_base.set(PSP_DEFAULTS.get(code, {}).get("PIX_API_BASE", ""))

    cbo.bind("<<ComboboxSelected>>", _update_fields)
    cbo_mode.bind("<<ComboboxSelected>>", _update_fields)
    _update_fields()

    def _save():
        _provider_code_only(); _mode_code_only()
        data = {
            "PIX_PROVIDER": v_provider.get().strip(),
            "PIX_KEY": v_key.get().strip(),
            "PIX_MERCHANT_NAME": v_name.get().strip(),
            "PIX_MERCHANT_CITY": v_city.get().strip().upper(),
            "PIX_API_BASE": v_api_base.get().strip(),
            "PIX_CLIENT_ID": v_client_id.get().strip(),
            "PIX_CLIENT_SECRET": v_client_secret.get().strip(),
            "PIX_CERT_PATH": v_cert_path.get().strip(),
            "PIX_WEBHOOK_URL": v_webhook_url.get().strip(),
            "PIX_TXID_PREFIX": v_txid_prefix.get().strip()[:12],
            "PIX_POLL_INTERVAL": v_poll_s.get().strip(),
            "PIX_CONFIRM_MODE": v_confirm.get().strip(),
            "PIX_WEBHOOK_PORT": v_webhook_port.get().strip(),
            "PIX_WEBHOOK_TOKEN": v_webhook_token.get().strip(),
        }
        with get_conn() as conn:
            _set_settings(conn, data)
        try:
            root.event_generate("<<PixSettingsChanged>>", when="tail")
        except Exception:
            pass
        messagebox.showinfo("PIX", "Configurações salvas.")

    def _pick_file(var: tk.StringVar):
        path = filedialog.askopenfilename(
            title="Selecionar certificado (PEM/P12)",
            filetypes=[("Certificados", "*.pem *.crt *.pfx *.p12"), ("Todos os arquivos", "*.*")]
        )
        if path:
            var.set(path)
